-- StudyPanda Database Schema
-- MySQL Database Tables for cPanel hosting
-- NOTE: Database should be pre-created in cPanel before running these queries

-- Users table
CREATE TABLE IF NOT EXISTS users (
    id INT PRIMARY KEY AUTO_INCREMENT,
    username VARCHAR(50) UNIQUE NOT NULL,
    email VARCHAR(255),
    password VARCHAR(255) NOT NULL,
    name VARCHAR(100) NOT NULL,
    age INT DEFAULT 6,
    avatar VARCHAR(50) DEFAULT '🐼',
    class_grade INT DEFAULT 1,
    date_of_birth DATE,
    parent_email VARCHAR(255),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    last_login TIMESTAMP,
    is_active BOOLEAN DEFAULT TRUE,
    is_premium BOOLEAN DEFAULT FALSE,
    INDEX idx_username (username),
    INDEX idx_email (email),
    INDEX idx_class (class_grade)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- User stats table
CREATE TABLE IF NOT EXISTS user_stats (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    coins INT DEFAULT 0,
    xp INT DEFAULT 0,
    level INT DEFAULT 1,
    streak_current INT DEFAULT 0,
    streak_best INT DEFAULT 0,
    streak_last_date DATE,
    total_games_played INT DEFAULT 0,
    total_correct_answers INT DEFAULT 0,
    total_wrong_answers INT DEFAULT 0,
    total_time_played INT DEFAULT 0,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_user (user_id),
    INDEX idx_level (level),
    INDEX idx_xp (xp)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Game progress table
CREATE TABLE IF NOT EXISTS game_progress (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    game_type VARCHAR(50) NOT NULL,
    game_id VARCHAR(50) NOT NULL,
    world_id INT DEFAULT 1,
    level_reached INT DEFAULT 1,
    high_score INT DEFAULT 0,
    stars_earned INT DEFAULT 0,
    times_played INT DEFAULT 0,
    last_played TIMESTAMP,
    completed BOOLEAN DEFAULT FALSE,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    UNIQUE KEY unique_user_game (user_id, game_type, game_id),
    INDEX idx_user_game (user_id, game_type)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Achievements table
CREATE TABLE IF NOT EXISTS achievements (
    id INT PRIMARY KEY AUTO_INCREMENT,
    code VARCHAR(50) UNIQUE NOT NULL,
    name VARCHAR(100) NOT NULL,
    description VARCHAR(255),
    emoji VARCHAR(20) NOT NULL,
    category VARCHAR(50),
    requirement_type VARCHAR(50),
    requirement_value INT DEFAULT 1,
    xp_reward INT DEFAULT 0,
    coin_reward INT DEFAULT 0,
    is_secret BOOLEAN DEFAULT FALSE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- User achievements (unlocked)
CREATE TABLE IF NOT EXISTS user_achievements (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    achievement_id INT NOT NULL,
    unlocked_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (achievement_id) REFERENCES achievements(id) ON DELETE CASCADE,
    UNIQUE KEY unique_user_achievement (user_id, achievement_id),
    INDEX idx_user (user_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Daily missions table
CREATE TABLE IF NOT EXISTS daily_missions (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    mission_date DATE NOT NULL,
    mission_type VARCHAR(50) NOT NULL,
    mission_title VARCHAR(100) NOT NULL,
    target_value INT NOT NULL,
    current_value INT DEFAULT 0,
    reward_type VARCHAR(50),
    reward_value INT,
    completed BOOLEAN DEFAULT FALSE,
    claimed BOOLEAN DEFAULT FALSE,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    UNIQUE KEY unique_user_date_mission (user_id, mission_date, mission_type),
    INDEX idx_user_date (user_id, mission_date)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Login rewards tracking
CREATE TABLE IF NOT EXISTS login_rewards (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    login_date DATE NOT NULL,
    day_number INT NOT NULL,
    reward_type VARCHAR(50),
    reward_value INT,
    claimed BOOLEAN DEFAULT FALSE,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    UNIQUE KEY unique_user_date (user_id, login_date),
    INDEX idx_user (user_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Friends table
CREATE TABLE IF NOT EXISTS friends (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    friend_id INT NOT NULL,
    status ENUM('pending', 'accepted', 'blocked') DEFAULT 'pending',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (friend_id) REFERENCES users(id) ON DELETE CASCADE,
    UNIQUE KEY unique_friendship (user_id, friend_id),
    INDEX idx_user (user_id),
    INDEX idx_friend (friend_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Teams/Clans table
CREATE TABLE IF NOT EXISTS teams (
    id INT PRIMARY KEY AUTO_INCREMENT,
    name VARCHAR(100) NOT NULL,
    emoji VARCHAR(20) DEFAULT '🐼',
    description VARCHAR(255),
    leader_id INT NOT NULL,
    level INT DEFAULT 1,
    xp INT DEFAULT 0,
    max_members INT DEFAULT 10,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (leader_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_leader (leader_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Team members
CREATE TABLE IF NOT EXISTS team_members (
    id INT PRIMARY KEY AUTO_INCREMENT,
    team_id INT NOT NULL,
    user_id INT NOT NULL,
    role ENUM('leader', 'co_leader', 'member') DEFAULT 'member',
    joined_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    contribution_xp INT DEFAULT 0,
    FOREIGN KEY (team_id) REFERENCES teams(id) ON DELETE CASCADE,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    UNIQUE KEY unique_team_user (team_id, user_id),
    INDEX idx_team (team_id),
    INDEX idx_user (user_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Challenges/Battles table
CREATE TABLE IF NOT EXISTS challenges (
    id INT PRIMARY KEY AUTO_INCREMENT,
    challenger_id INT NOT NULL,
    opponent_id INT NOT NULL,
    game_type VARCHAR(50) NOT NULL,
    challenger_score INT DEFAULT 0,
    opponent_score INT DEFAULT 0,
    winner_id INT,
    status ENUM('pending', 'active', 'completed', 'expired') DEFAULT 'pending',
    coins_reward INT DEFAULT 0,
    xp_reward INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    completed_at TIMESTAMP,
    FOREIGN KEY (challenger_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (opponent_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_challenger (challenger_id),
    INDEX idx_opponent (opponent_id),
    INDEX idx_status (status)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Leaderboard scores (cached daily)
CREATE TABLE IF NOT EXISTS leaderboard (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    period_type ENUM('daily', 'weekly', 'monthly', 'alltime') NOT NULL,
    period_date DATE NOT NULL,
    score INT DEFAULT 0,
    rank_position INT,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    UNIQUE KEY unique_user_period (user_id, period_type, period_date),
    INDEX idx_period_score (period_type, period_date, score DESC)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Power-ups inventory
CREATE TABLE IF NOT EXISTS user_powerups (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    powerup_type VARCHAR(50) NOT NULL,
    quantity INT DEFAULT 0,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    UNIQUE KEY unique_user_powerup (user_id, powerup_type),
    INDEX idx_user (user_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Activity log for analytics
CREATE TABLE IF NOT EXISTS activity_log (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    action_type VARCHAR(50) NOT NULL,
    action_data JSON,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_user_time (user_id, created_at),
    INDEX idx_action (action_type)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Parent controls
CREATE TABLE IF NOT EXISTS parent_settings (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    parent_pin VARCHAR(10),
    daily_time_limit INT DEFAULT 60,
    enabled_days VARCHAR(50) DEFAULT '1,2,3,4,5,6,7',
    start_time TIME DEFAULT '08:00:00',
    end_time TIME DEFAULT '20:00:00',
    notifications_enabled BOOLEAN DEFAULT TRUE,
    weekly_report_enabled BOOLEAN DEFAULT TRUE,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    UNIQUE KEY unique_user (user_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Insert default achievements
INSERT INTO achievements (code, name, description, emoji, category, requirement_type, requirement_value, xp_reward, coin_reward) VALUES
('first_game', 'First Steps', 'Play your first game', '🎮', 'beginner', 'games_played', 1, 50, 25),
('streak_3', 'On Fire', 'Maintain a 3 day streak', '🔥', 'streak', 'streak_days', 3, 100, 50),
('streak_7', 'Week Warrior', 'Maintain a 7 day streak', '⚡', 'streak', 'streak_days', 7, 250, 100),
('streak_30', 'Monthly Master', 'Maintain a 30 day streak', '🌟', 'streak', 'streak_days', 30, 1000, 500),
('level_5', 'Rising Star', 'Reach level 5', '⭐', 'level', 'level', 5, 150, 75),
('level_10', 'Super Student', 'Reach level 10', '🌟', 'level', 'level', 10, 300, 150),
('level_25', 'Genius', 'Reach level 25', '🧠', 'level', 'level', 25, 750, 400),
('perfect_10', 'Perfect Ten', 'Get 10 perfect scores', '💯', 'score', 'perfect_scores', 10, 200, 100),
('coins_1000', 'Rich Panda', 'Earn 1000 coins', '💰', 'coins', 'total_coins', 1000, 100, 0),
('math_master', 'Math Wizard', 'Complete all Math games', '🔢', 'subject', 'math_complete', 1, 500, 250),
('science_pro', 'Science Explorer', 'Complete all Science games', '🔬', 'subject', 'science_complete', 1, 500, 250),
('brain_champ', 'Brain Champion', 'Complete all Brain games', '🧩', 'subject', 'brain_complete', 1, 500, 250),
('friend_5', 'Social Butterfly', 'Make 5 friends', '👥', 'social', 'friends_count', 5, 150, 75),
('team_player', 'Team Player', 'Join a team', '🤝', 'social', 'team_joined', 1, 100, 50),
('challenge_win', 'Champion', 'Win your first challenge', '🏆', 'pvp', 'challenges_won', 1, 150, 75);
