<?php
/**
 * User API Endpoints
 */

function handleUser($action, $method, $id) {
    $user = JWT::requireAuth();
    
    switch ($action) {
        case 'profile':
            if ($method === 'GET') getProfile($user);
            elseif ($method === 'PUT') updateProfile($user);
            else Response::error('Method not allowed', 405);
            break;
        case 'stats':
            if ($method === 'GET') getStats($user);
            elseif ($method === 'PUT') updateStats($user);
            else Response::error('Method not allowed', 405);
            break;
        case 'avatar':
            if ($method === 'PUT') updateAvatar($user);
            else Response::error('Method not allowed', 405);
            break;
        case 'powerups':
            if ($method === 'GET') getPowerups($user);
            elseif ($method === 'POST') usePowerup($user);
            else Response::error('Method not allowed', 405);
            break;
        default:
            Response::error('Endpoint not found', 404);
    }
}

function getProfile($authUser) {
    $db = Database::getInstance()->getConnection();
    
    $stmt = $db->prepare("
        SELECT u.id, u.name, u.email, u.avatar, u.class_grade, u.is_premium,
               us.coins, us.xp, us.level, us.streak_current, us.streak_best,
               us.total_games_played, us.total_correct_answers, us.total_time_played
        FROM users u
        LEFT JOIN user_stats us ON u.id = us.user_id
        WHERE u.id = ?
    ");
    $stmt->execute([$authUser['user_id']]);
    $user = $stmt->fetch();
    
    if (!$user) {
        Response::error('User not found', 404);
    }
    
    Response::success($user);
}

function updateProfile($authUser) {
    $data = Request::getJSON();
    $db = Database::getInstance()->getConnection();
    
    $allowedFields = ['name', 'class_grade', 'date_of_birth', 'parent_email'];
    $updates = [];
    $params = [];
    
    foreach ($allowedFields as $field) {
        if (isset($data[$field])) {
            $updates[] = "$field = ?";
            $params[] = $data[$field];
        }
    }
    
    if (empty($updates)) {
        Response::error('No valid fields to update');
    }
    
    $params[] = $authUser['user_id'];
    $sql = "UPDATE users SET " . implode(', ', $updates) . " WHERE id = ?";
    
    $stmt = $db->prepare($sql);
    $stmt->execute($params);
    
    Response::success(null, 'Profile updated successfully');
}

function getStats($authUser) {
    $db = Database::getInstance()->getConnection();
    
    $stmt = $db->prepare("SELECT * FROM user_stats WHERE user_id = ?");
    $stmt->execute([$authUser['user_id']]);
    $stats = $stmt->fetch();
    
    Response::success($stats);
}

function updateStats($authUser) {
    $data = Request::getJSON();
    $db = Database::getInstance()->getConnection();
    
    $coinsToAdd = $data['coins'] ?? 0;
    $xpToAdd = $data['xp'] ?? 0;
    $gamesPlayed = $data['games_played'] ?? 0;
    $correctAnswers = $data['correct_answers'] ?? 0;
    $wrongAnswers = $data['wrong_answers'] ?? 0;
    $timePlayed = $data['time_played'] ?? 0;
    
    // Update stats
    $stmt = $db->prepare("
        UPDATE user_stats SET
            coins = coins + ?,
            xp = xp + ?,
            total_games_played = total_games_played + ?,
            total_correct_answers = total_correct_answers + ?,
            total_wrong_answers = total_wrong_answers + ?,
            total_time_played = total_time_played + ?
        WHERE user_id = ?
    ");
    $stmt->execute([
        $coinsToAdd, $xpToAdd, $gamesPlayed, 
        $correctAnswers, $wrongAnswers, $timePlayed,
        $authUser['user_id']
    ]);
    
    // Check for level up
    $stmt = $db->prepare("SELECT xp, level FROM user_stats WHERE user_id = ?");
    $stmt->execute([$authUser['user_id']]);
    $stats = $stmt->fetch();
    
    $newLevel = calculateLevel($stats['xp']);
    $leveledUp = $newLevel > $stats['level'];
    
    if ($leveledUp) {
        $stmt = $db->prepare("UPDATE user_stats SET level = ? WHERE user_id = ?");
        $stmt->execute([$newLevel, $authUser['user_id']]);
    }
    
    Response::success([
        'level' => $newLevel,
        'leveled_up' => $leveledUp,
        'xp' => $stats['xp']
    ], 'Stats updated');
}

function calculateLevel($xp) {
    // XP required per level: 100, 200, 300, 400...
    $level = 1;
    $xpRequired = 0;
    while ($xp >= $xpRequired + ($level * 100)) {
        $xpRequired += $level * 100;
        $level++;
    }
    return $level;
}

function updateAvatar($authUser) {
    $data = Request::getJSON();
    $avatar = $data['avatar'] ?? '🐼';
    
    $db = Database::getInstance()->getConnection();
    $stmt = $db->prepare("UPDATE users SET avatar = ? WHERE id = ?");
    $stmt->execute([$avatar, $authUser['user_id']]);
    
    Response::success(['avatar' => $avatar], 'Avatar updated');
}

function getPowerups($authUser) {
    $db = Database::getInstance()->getConnection();
    
    $stmt = $db->prepare("SELECT powerup_type, quantity FROM user_powerups WHERE user_id = ?");
    $stmt->execute([$authUser['user_id']]);
    $powerups = $stmt->fetchAll();
    
    Response::success($powerups);
}

function usePowerup($authUser) {
    $data = Request::getJSON();
    $powerupType = $data['powerup_type'] ?? '';
    
    if (empty($powerupType)) {
        Response::error('Power-up type is required');
    }
    
    $db = Database::getInstance()->getConnection();
    
    // Check if user has the powerup
    $stmt = $db->prepare("SELECT quantity FROM user_powerups WHERE user_id = ? AND powerup_type = ?");
    $stmt->execute([$authUser['user_id'], $powerupType]);
    $powerup = $stmt->fetch();
    
    if (!$powerup || $powerup['quantity'] < 1) {
        Response::error('Power-up not available');
    }
    
    // Use powerup
    $stmt = $db->prepare("UPDATE user_powerups SET quantity = quantity - 1 WHERE user_id = ? AND powerup_type = ?");
    $stmt->execute([$authUser['user_id'], $powerupType]);
    
    Response::success(['remaining' => $powerup['quantity'] - 1], 'Power-up used');
}
?>
