<?php
/**
 * Tournament API Endpoints
 * Handles tournament creation, brackets, and progression
 */

require_once '../config.php';

function handleTournament($action, $method) {
    switch ($action) {
        case 'create':
            if ($method === 'POST') createTournament();
            break;
        case 'list':
            if ($method === 'GET') listTournaments();
            break;
        case 'join':
            if ($method === 'POST') joinTournament();
            break;
        case 'bracket':
            if ($method === 'GET') getTournamentBracket();
            break;
        case 'match_result':
            if ($method === 'POST') recordMatchResult();
            break;
        case 'leaderboard':
            if ($method === 'GET') getTournamentLeaderboard();
            break;
        default:
            Response::error('Invalid action', 400);
    }
}

/**
 * Create a new tournament
 */
function createTournament() {
    $user = JWT::requireAuth();
    $db = Database::getInstance()->getConnection();
    
    $name = Request::post('name', 'Tournament');
    $gameType = Request::post('game_type');
    $entryFee = Request::post('entry_fee', 10);
    $maxPlayers = Request::post('max_players', 8); // Must be power of 2
    $startTime = Request::post('start_time'); // Unix timestamp
    $prizePool = Request::post('prize_pool', 0);
    
    // Validate max players (must be 4, 8, 16, 32)
    $validSizes = [4, 8, 16, 32];
    if (!in_array($maxPlayers, $validSizes)) {
        Response::error('Max players must be 4, 8, 16, or 32', 400);
    }
    
    $stmt = $db->prepare("
        INSERT INTO tournaments 
        (name, game_type, entry_fee, max_players, start_time, prize_pool, creator_id, status)
        VALUES (?, ?, ?, ?, FROM_UNIXTIME(?), ?, ?, 'registration')
    ");
    $stmt->execute([$name, $gameType, $entryFee, $maxPlayers, $startTime, $prizePool, $user['user_id']]);
    $tournamentId = $db->lastInsertId();
    
    Response::success([
        'tournament_id' => $tournamentId,
        'name' => $name,
        'max_players' => $maxPlayers
    ], 'Tournament created!');
}

/**
 * List active/upcoming tournaments
 */
function listTournaments() {
    $db = Database::getInstance()->getConnection();
    $gameType = Request::get('game_type');
    $status = Request::get('status', 'registration');
    
    $sql = "
        SELECT t.*, 
               u.name as creator_name,
               (SELECT COUNT(*) FROM tournament_players WHERE tournament_id = t.id) as current_players
        FROM tournaments t
        JOIN users u ON t.creator_id = u.id
        WHERE t.status = ?
    ";
    $params = [$status];
    
    if ($gameType) {
        $sql .= " AND t.game_type = ?";
        $params[] = $gameType;
    }
    
    $sql .= " ORDER BY t.start_time ASC LIMIT 50";
    
    $stmt = $db->prepare($sql);
    $stmt->execute($params);
    $tournaments = $stmt->fetchAll();
    
    Response::success(['tournaments' => $tournaments]);
}

/**
 * Join a tournament
 */
function joinTournament() {
    $user = JWT::requireAuth();
    $db = Database::getInstance()->getConnection();
    
    $tournamentId = Request::post('tournament_id');
    
    // Get tournament
    $stmt = $db->prepare("SELECT * FROM tournaments WHERE id = ?");
    $stmt->execute([$tournamentId]);
    $tournament = $stmt->fetch();
    
    if (!$tournament) {
        Response::error('Tournament not found', 404);
    }
    
    if ($tournament['status'] !== 'registration') {
        Response::error('Tournament registration closed', 400);
    }
    
    // Check if already joined
    $stmt = $db->prepare("SELECT id FROM tournament_players WHERE tournament_id = ? AND user_id = ?");
    $stmt->execute([$tournamentId, $user['user_id']]);
    if ($stmt->fetch()) {
        Response::error('Already joined', 400);
    }
    
    // Check player count
    $stmt = $db->prepare("SELECT COUNT(*) as count FROM tournament_players WHERE tournament_id = ?");
    $stmt->execute([$tournamentId]);
    $count = $stmt->fetch()['count'];
    
    if ($count >= $tournament['max_players']) {
        Response::error('Tournament full', 400);
    }
    
    // Check coins
    $stmt = $db->prepare("SELECT coins FROM user_stats WHERE user_id = ?");
    $stmt->execute([$user['user_id']]);
    $stats = $stmt->fetch();
    
    if ($stats['coins'] < $tournament['entry_fee']) {
        Response::error('Insufficient coins', 400);
    }
    
    // Deduct entry fee and join
    $db->prepare("UPDATE user_stats SET coins = coins - ? WHERE user_id = ?")
       ->execute([$tournament['entry_fee'], $user['user_id']]);
    
    // Add to prize pool
    $db->prepare("UPDATE tournaments SET prize_pool = prize_pool + ? WHERE id = ?")
       ->execute([$tournament['entry_fee'], $tournamentId]);
    
    $stmt = $db->prepare("
        INSERT INTO tournament_players (tournament_id, user_id, seed, status)
        VALUES (?, ?, ?, 'active')
    ");
    $stmt->execute([$tournamentId, $user['user_id'], $count + 1]);
    
    // Check if tournament is full - start it
    if ($count + 1 >= $tournament['max_players']) {
        generateBracket($tournamentId, $tournament['max_players']);
        $db->prepare("UPDATE tournaments SET status = 'active' WHERE id = ?")->execute([$tournamentId]);
    }
    
    Response::success([
        'joined' => true,
        'seed' => $count + 1,
        'players_registered' => $count + 1,
        'max_players' => $tournament['max_players']
    ]);
}

/**
 * Generate tournament bracket
 */
function generateBracket($tournamentId, $playerCount) {
    $db = Database::getInstance()->getConnection();
    
    // Get players with random seed
    $stmt = $db->prepare("
        SELECT user_id FROM tournament_players 
        WHERE tournament_id = ? 
        ORDER BY RAND()
    ");
    $stmt->execute([$tournamentId]);
    $players = $stmt->fetchAll(PDO::FETCH_COLUMN);
    
    // Update seeds
    foreach ($players as $index => $userId) {
        $db->prepare("UPDATE tournament_players SET seed = ? WHERE tournament_id = ? AND user_id = ?")
           ->execute([$index + 1, $tournamentId, $userId]);
    }
    
    // Create first round matches
    $round = 1;
    $matchesInRound = $playerCount / 2;
    
    for ($i = 0; $i < $matchesInRound; $i++) {
        $player1 = $players[$i * 2];
        $player2 = $players[$i * 2 + 1];
        
        $db->prepare("
            INSERT INTO tournament_matches 
            (tournament_id, round, match_number, player1_id, player2_id, status)
            VALUES (?, ?, ?, ?, ?, 'pending')
        ")->execute([$tournamentId, $round, $i + 1, $player1, $player2]);
    }
}

/**
 * Get tournament bracket
 */
function getTournamentBracket() {
    $db = Database::getInstance()->getConnection();
    $tournamentId = Request::get('tournament_id');
    
    $stmt = $db->prepare("
        SELECT tm.*,
               u1.name as player1_name, u1.avatar as player1_avatar,
               u2.name as player2_name, u2.avatar as player2_avatar,
               uw.name as winner_name
        FROM tournament_matches tm
        LEFT JOIN users u1 ON tm.player1_id = u1.id
        LEFT JOIN users u2 ON tm.player2_id = u2.id
        LEFT JOIN users uw ON tm.winner_id = uw.id
        WHERE tm.tournament_id = ?
        ORDER BY tm.round, tm.match_number
    ");
    $stmt->execute([$tournamentId]);
    $matches = $stmt->fetchAll();
    
    // Group by round
    $bracket = [];
    foreach ($matches as $match) {
        $round = $match['round'];
        if (!isset($bracket[$round])) {
            $bracket[$round] = [];
        }
        $bracket[$round][] = $match;
    }
    
    Response::success(['bracket' => $bracket]);
}

/**
 * Record match result and advance winner
 */
function recordMatchResult() {
    $user = JWT::requireAuth();
    $db = Database::getInstance()->getConnection();
    
    $matchId = Request::post('match_id');
    $winnerId = Request::post('winner_id');
    $player1Score = Request::post('player1_score', 0);
    $player2Score = Request::post('player2_score', 0);
    
    // Get match
    $stmt = $db->prepare("SELECT * FROM tournament_matches WHERE id = ?");
    $stmt->execute([$matchId]);
    $match = $stmt->fetch();
    
    if (!$match || $match['status'] !== 'active') {
        Response::error('Invalid match', 400);
    }
    
    // Update match
    $db->prepare("
        UPDATE tournament_matches 
        SET winner_id = ?, player1_score = ?, player2_score = ?, 
            status = 'completed', completed_at = NOW()
        WHERE id = ?
    ")->execute([$winnerId, $player1Score, $player2Score, $matchId]);
    
    // Update player status
    $loserId = ($winnerId == $match['player1_id']) ? $match['player2_id'] : $match['player1_id'];
    $db->prepare("UPDATE tournament_players SET status = 'eliminated' WHERE tournament_id = ? AND user_id = ?")
       ->execute([$match['tournament_id'], $loserId]);
    
    // Check if there's a next round match
    advanceWinner($match['tournament_id'], $match['round'], $match['match_number'], $winnerId);
    
    Response::success(['winner_advanced' => true]);
}

/**
 * Advance winner to next round
 */
function advanceWinner($tournamentId, $currentRound, $matchNumber, $winnerId) {
    $db = Database::getInstance()->getConnection();
    
    $nextRound = $currentRound + 1;
    $nextMatchNumber = ceil($matchNumber / 2);
    $isPlayer1 = ($matchNumber % 2 == 1);
    
    // Check if next match exists
    $stmt = $db->prepare("
        SELECT id FROM tournament_matches 
        WHERE tournament_id = ? AND round = ? AND match_number = ?
    ");
    $stmt->execute([$tournamentId, $nextRound, $nextMatchNumber]);
    $nextMatch = $stmt->fetch();
    
    if ($nextMatch) {
        // Update existing match
        $field = $isPlayer1 ? 'player1_id' : 'player2_id';
        $db->prepare("UPDATE tournament_matches SET $field = ? WHERE id = ?")
           ->execute([$winnerId, $nextMatch['id']]);
    } else {
        // Check if this was the final
        $stmt = $db->prepare("
            SELECT COUNT(*) as remaining FROM tournament_players 
            WHERE tournament_id = ? AND status = 'active'
        ");
        $stmt->execute([$tournamentId]);
        $remaining = $stmt->fetch()['remaining'];
        
        if ($remaining <= 1) {
            // Tournament complete!
            $db->prepare("UPDATE tournaments SET status = 'completed', winner_id = ? WHERE id = ?")
               ->execute([$winnerId, $tournamentId]);
            
            // Award prize pool
            $stmt = $db->prepare("SELECT prize_pool FROM tournaments WHERE id = ?");
            $stmt->execute([$tournamentId]);
            $prizePool = $stmt->fetch()['prize_pool'];
            
            $db->prepare("UPDATE user_stats SET coins = coins + ? WHERE user_id = ?")
               ->execute([$prizePool, $winnerId]);
        } else {
            // Create next round match
            $db->prepare("
                INSERT INTO tournament_matches 
                (tournament_id, round, match_number, player1_id, status)
                VALUES (?, ?, ?, ?, 'pending')
            ")->execute([$tournamentId, $nextRound, $nextMatchNumber, $winnerId]);
        }
    }
}

/**
 * Get tournament leaderboard
 */
function getTournamentLeaderboard() {
    $db = Database::getInstance()->getConnection();
    $tournamentId = Request::get('tournament_id');
    
    $stmt = $db->prepare("
        SELECT tp.*, u.name, u.avatar,
               (SELECT MAX(round) FROM tournament_matches 
                WHERE tournament_id = tp.tournament_id 
                  AND (player1_id = tp.user_id OR player2_id = tp.user_id)
                  AND winner_id = tp.user_id) as rounds_won
        FROM tournament_players tp
        JOIN users u ON tp.user_id = u.id
        WHERE tp.tournament_id = ?
        ORDER BY 
            CASE WHEN tp.status = 'active' THEN 0 ELSE 1 END,
            rounds_won DESC
    ");
    $stmt->execute([$tournamentId]);
    $leaderboard = $stmt->fetchAll();
    
    Response::success(['leaderboard' => $leaderboard]);
}
?>
