<?php
/**
 * Progress & Sync API Endpoints
 */

function handleProgress($action, $method) {
    $user = JWT::requireAuth();
    
    switch ($action) {
        case 'game':
            if ($method === 'GET') getGameProgress($user);
            elseif ($method === 'POST') saveGameProgress($user);
            else Response::error('Method not allowed', 405);
            break;
        case 'all':
            if ($method === 'GET') getAllProgress($user);
            else Response::error('Method not allowed', 405);
            break;
        default:
            Response::error('Endpoint not found', 404);
    }
}

function getGameProgress($user) {
    $gameType = Request::get('game_type');
    $gameId = Request::get('game_id');
    
    $db = Database::getInstance()->getConnection();
    
    if ($gameType && $gameId) {
        $stmt = $db->prepare("
            SELECT * FROM game_progress 
            WHERE user_id = ? AND game_type = ? AND game_id = ?
        ");
        $stmt->execute([$user['user_id'], $gameType, $gameId]);
    } else {
        $stmt = $db->prepare("SELECT * FROM game_progress WHERE user_id = ?");
        $stmt->execute([$user['user_id']]);
    }
    
    $progress = $gameType && $gameId ? $stmt->fetch() : $stmt->fetchAll();
    Response::success($progress);
}

function saveGameProgress($user) {
    $data = Request::getJSON();
    
    $gameType = $data['game_type'] ?? '';
    $gameId = $data['game_id'] ?? '';
    $score = $data['score'] ?? 0;
    $stars = $data['stars'] ?? 0;
    $level = $data['level_reached'] ?? 1;
    $completed = $data['completed'] ?? false;
    $coinsEarned = $data['coins_earned'] ?? 0;
    $xpEarned = $data['xp_earned'] ?? 0;
    $correctAnswers = $data['correct_answers'] ?? 0;
    $wrongAnswers = $data['wrong_answers'] ?? 0;
    
    if (empty($gameType) || empty($gameId)) {
        Response::error('Game type and ID are required');
    }
    
    $db = Database::getInstance()->getConnection();
    
    // Upsert game progress
    $stmt = $db->prepare("
        INSERT INTO game_progress (user_id, game_type, game_id, level_reached, high_score, stars_earned, times_played, completed, last_played)
        VALUES (?, ?, ?, ?, ?, ?, 1, ?, NOW())
        ON DUPLICATE KEY UPDATE
            level_reached = GREATEST(level_reached, VALUES(level_reached)),
            high_score = GREATEST(high_score, VALUES(high_score)),
            stars_earned = GREATEST(stars_earned, VALUES(stars_earned)),
            times_played = times_played + 1,
            completed = completed OR VALUES(completed),
            last_played = NOW()
    ");
    $stmt->execute([$user['user_id'], $gameType, $gameId, $level, $score, $stars, $completed]);
    
    // Update user stats
    $stmt = $db->prepare("
        UPDATE user_stats SET
            coins = coins + ?,
            xp = xp + ?,
            total_games_played = total_games_played + 1,
            total_correct_answers = total_correct_answers + ?,
            total_wrong_answers = total_wrong_answers + ?
        WHERE user_id = ?
    ");
    $stmt->execute([$coinsEarned, $xpEarned, $correctAnswers, $wrongAnswers, $user['user_id']]);
    
    // Log activity
    $stmt = $db->prepare("
        INSERT INTO activity_log (user_id, action_type, action_data)
        VALUES (?, 'game_complete', ?)
    ");
    $stmt->execute([$user['user_id'], json_encode([
        'game_type' => $gameType,
        'game_id' => $gameId,
        'score' => $score,
        'coins' => $coinsEarned,
        'xp' => $xpEarned
    ])]);
    
    // Check level up
    $stmt = $db->prepare("SELECT xp, level FROM user_stats WHERE user_id = ?");
    $stmt->execute([$user['user_id']]);
    $stats = $stmt->fetch();
    
    $newLevel = calculateLevel($stats['xp']);
    if ($newLevel > $stats['level']) {
        $stmt = $db->prepare("UPDATE user_stats SET level = ? WHERE user_id = ?");
        $stmt->execute([$newLevel, $user['user_id']]);
    }
    
    Response::success([
        'new_level' => $newLevel,
        'total_xp' => $stats['xp'],
        'leveled_up' => $newLevel > $stats['level']
    ], 'Progress saved');
}

function getAllProgress($user) {
    $db = Database::getInstance()->getConnection();
    
    // Game progress
    $stmt = $db->prepare("SELECT * FROM game_progress WHERE user_id = ?");
    $stmt->execute([$user['user_id']]);
    $gameProgress = $stmt->fetchAll();
    
    // User stats
    $stmt = $db->prepare("SELECT * FROM user_stats WHERE user_id = ?");
    $stmt->execute([$user['user_id']]);
    $stats = $stmt->fetch();
    
    // Achievements
    $stmt = $db->prepare("
        SELECT a.* FROM achievements a
        INNER JOIN user_achievements ua ON a.id = ua.achievement_id
        WHERE ua.user_id = ?
    ");
    $stmt->execute([$user['user_id']]);
    $achievements = $stmt->fetchAll();
    
    Response::success([
        'game_progress' => $gameProgress,
        'stats' => $stats,
        'achievements' => $achievements
    ]);
}

function handleSync($method) {
    if ($method !== 'POST') {
        Response::error('Method not allowed', 405);
    }
    
    $user = JWT::requireAuth();
    $data = Request::getJSON();
    
    $localProgress = $data['progress'] ?? [];
    $localStats = $data['stats'] ?? [];
    $lastSyncTime = $data['last_sync'] ?? 0;
    
    $db = Database::getInstance()->getConnection();
    
    // Get server data
    $stmt = $db->prepare("SELECT * FROM game_progress WHERE user_id = ?");
    $stmt->execute([$user['user_id']]);
    $serverProgress = $stmt->fetchAll();
    
    $stmt = $db->prepare("SELECT * FROM user_stats WHERE user_id = ?");
    $stmt->execute([$user['user_id']]);
    $serverStats = $stmt->fetch();
    
    // Merge logic: keep highest scores, combine totals
    foreach ($localProgress as $local) {
        $found = false;
        foreach ($serverProgress as &$server) {
            if ($server['game_type'] === $local['game_type'] && $server['game_id'] === $local['game_id']) {
                $found = true;
                // Merge: keep highest
                $stmt = $db->prepare("
                    UPDATE game_progress SET
                        high_score = GREATEST(high_score, ?),
                        stars_earned = GREATEST(stars_earned, ?),
                        level_reached = GREATEST(level_reached, ?),
                        times_played = GREATEST(times_played, ?),
                        completed = completed OR ?
                    WHERE user_id = ? AND game_type = ? AND game_id = ?
                ");
                $stmt->execute([
                    $local['high_score'] ?? 0,
                    $local['stars_earned'] ?? 0,
                    $local['level_reached'] ?? 1,
                    $local['times_played'] ?? 0,
                    $local['completed'] ?? false,
                    $user['user_id'],
                    $local['game_type'],
                    $local['game_id']
                ]);
                break;
            }
        }
        
        if (!$found) {
            // Insert new
            $stmt = $db->prepare("
                INSERT INTO game_progress (user_id, game_type, game_id, high_score, stars_earned, level_reached, times_played, completed)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?)
            ");
            $stmt->execute([
                $user['user_id'],
                $local['game_type'],
                $local['game_id'],
                $local['high_score'] ?? 0,
                $local['stars_earned'] ?? 0,
                $local['level_reached'] ?? 1,
                $local['times_played'] ?? 0,
                $local['completed'] ?? false
            ]);
        }
    }
    
    // Sync stats: keep highest values
    if (!empty($localStats)) {
        $stmt = $db->prepare("
            UPDATE user_stats SET
                coins = GREATEST(coins, ?),
                xp = GREATEST(xp, ?),
                level = GREATEST(level, ?),
                streak_best = GREATEST(streak_best, ?)
            WHERE user_id = ?
        ");
        $stmt->execute([
            $localStats['coins'] ?? 0,
            $localStats['xp'] ?? 0,
            $localStats['level'] ?? 1,
            $localStats['streak_best'] ?? 0,
            $user['user_id']
        ]);
    }
    
    // Return merged data
    $stmt = $db->prepare("SELECT * FROM game_progress WHERE user_id = ?");
    $stmt->execute([$user['user_id']]);
    $mergedProgress = $stmt->fetchAll();
    
    $stmt = $db->prepare("SELECT * FROM user_stats WHERE user_id = ?");
    $stmt->execute([$user['user_id']]);
    $mergedStats = $stmt->fetch();
    
    Response::success([
        'progress' => $mergedProgress,
        'stats' => $mergedStats,
        'sync_time' => time()
    ], 'Data synced successfully');
}

function calculateLevel($xp) {
    $level = 1;
    $xpRequired = 0;
    while ($xp >= $xpRequired + ($level * 100)) {
        $xpRequired += $level * 100;
        $level++;
    }
    return $level;
}
?>
