<?php
/**
 * Authentication API Endpoints
 */

function handleAuth($action, $method) {
    switch ($action) {
        case 'register':
            if ($method !== 'POST') Response::error('Method not allowed', 405);
            register();
            break;
        case 'login':
            if ($method !== 'POST') Response::error('Method not allowed', 405);
            login();
            break;
        case 'logout':
            logout();
            break;
        case 'forgot-password':
            if ($method !== 'POST') Response::error('Method not allowed', 405);
            forgotPassword();
            break;
        case 'verify':
            verifyToken();
            break;
        default:
            Response::error('Endpoint not found', 404);
    }
}

function register() {
    $data = Request::getJSON();
    
    // Validate input
    $username = $data['username'] ?? '';
    $password = $data['password'] ?? '';
    $name = $data['name'] ?? '';
    $age = $data['age'] ?? 6;
    $email = $data['email'] ?? null; // Optional
    $classGrade = $data['class_grade'] ?? 1;
    
    // Validate username (alphanumeric, 3-20 chars)
    if (!Validator::required($username) || !Validator::minLength($username, 3)) {
        Response::error('Username is required (min 3 characters)');
    }
    if (!preg_match('/^[a-zA-Z0-9_]+$/', $username)) {
        Response::error('Username can only contain letters, numbers and underscore');
    }
    if (!Validator::required($name) || !Validator::minLength($name, 2)) {
        Response::error('Name is required (min 2 characters)');
    }
    if (!Validator::minLength($password, 4)) {
        Response::error('Password must be at least 4 characters');
    }
    // Email is optional, but if provided must be valid
    if (!empty($email) && !Validator::email($email)) {
        Response::error('Please enter a valid email address');
    }
    
    $db = Database::getInstance()->getConnection();
    
    // Check if username exists
    $stmt = $db->prepare("SELECT id FROM users WHERE username = ?");
    $stmt->execute([$username]);
    if ($stmt->fetch()) {
        Response::error('Username already taken. Please choose another one.');
    }
    
    // Check if email exists (if provided)
    if (!empty($email)) {
        $stmt = $db->prepare("SELECT id FROM users WHERE email = ?");
        $stmt->execute([$email]);
        if ($stmt->fetch()) {
            Response::error('Email already registered');
        }
    }
    
    // Create user
    $hashedPassword = password_hash($password, PASSWORD_BCRYPT);
    
    $stmt = $db->prepare("INSERT INTO users (username, name, email, password, age, class_grade) VALUES (?, ?, ?, ?, ?, ?)");
    $stmt->execute([$username, $name, $email, $hashedPassword, $age, $classGrade]);
    $userId = $db->lastInsertId();
    
    // Create user stats
    $stmt = $db->prepare("INSERT INTO user_stats (user_id) VALUES (?)");
    $stmt->execute([$userId]);
    
    // Generate token
    $token = JWT::encode(['user_id' => $userId, 'username' => $username]);
    
    Response::success([
        'user' => [
            'id' => $userId,
            'username' => $username,
            'name' => $name,
            'email' => $email,
            'age' => $age,
            'avatar' => '🐼',
            'class_grade' => $classGrade,
            'coins' => 0,
            'xp' => 0,
            'level' => 1,
            'streak' => 0
        ],
        'token' => $token
    ], 'Registration successful! Welcome to StudyPanda!', 201);
}

function login() {
    $data = Request::getJSON();
    
    $username = $data['username'] ?? '';
    $password = $data['password'] ?? '';
    
    if (empty($username) || empty($password)) {
        Response::error('Username and password are required');
    }
    
    $db = Database::getInstance()->getConnection();
    
    $stmt = $db->prepare("
        SELECT u.*, us.coins, us.xp, us.level, us.streak_current 
        FROM users u 
        LEFT JOIN user_stats us ON u.id = us.user_id 
        WHERE u.username = ? AND u.is_active = 1
    ");
    $stmt->execute([$username]);
    $user = $stmt->fetch();
    
    if (!$user || !password_verify($password, $user['password'])) {
        Response::error('Invalid username or password', 401);
    }
    
    // Update last login
    $stmt = $db->prepare("UPDATE users SET last_login = NOW() WHERE id = ?");
    $stmt->execute([$user['id']]);
    
    // Update streak
    updateStreak($user['id']);
    
    // Generate token
    $token = JWT::encode(['user_id' => $user['id'], 'username' => $username]);
    
    Response::success([
        'user' => [
            'id' => $user['id'],
            'username' => $user['username'],
            'name' => $user['name'],
            'email' => $user['email'],
            'age' => $user['age'] ?? 6,
            'avatar' => $user['avatar'],
            'class_grade' => $user['class_grade'],
            'coins' => $user['coins'] ?? 0,
            'xp' => $user['xp'] ?? 0,
            'level' => $user['level'] ?? 1,
            'streak' => $user['streak_current'] ?? 0
        ],
        'token' => $token
    ], 'Welcome back!');
}

function logout() {
    Response::success(null, 'Logged out successfully');
}

function forgotPassword() {
    $data = Request::getJSON();
    $identifier = $data['username'] ?? $data['email'] ?? '';
    
    if (empty($identifier)) {
        Response::error('Username or email is required');
    }
    
    $db = Database::getInstance()->getConnection();
    
    // Find user by username or email
    $stmt = $db->prepare("SELECT id, email, username FROM users WHERE username = ? OR email = ?");
    $stmt->execute([$identifier, $identifier]);
    $user = $stmt->fetch();
    
    if (!$user) {
        Response::error('No account found with this username or email');
    }
    
    if (empty($user['email'])) {
        Response::error('No email associated with this account. Please contact support.');
    }
    
    // In production, send reset email here
    // For now, just return success
    Response::success([
        'email_hint' => substr($user['email'], 0, 3) . '***' . substr($user['email'], strpos($user['email'], '@'))
    ], 'Password reset instructions sent to your email');
}

function verifyToken() {
    $user = JWT::getAuthUser();
    if ($user) {
        Response::success(['valid' => true, 'user_id' => $user['user_id']]);
    } else {
        Response::error('Invalid or expired token', 401);
    }
}

function updateStreak($userId) {
    $db = Database::getInstance()->getConnection();
    
    $stmt = $db->prepare("SELECT streak_current, streak_best, streak_last_date FROM user_stats WHERE user_id = ?");
    $stmt->execute([$userId]);
    $stats = $stmt->fetch();
    
    $today = date('Y-m-d');
    $yesterday = date('Y-m-d', strtotime('-1 day'));
    $lastDate = $stats['streak_last_date'];
    
    $newStreak = $stats['streak_current'];
    
    if ($lastDate === $today) {
        // Already logged in today
        return;
    } elseif ($lastDate === $yesterday) {
        // Continue streak
        $newStreak++;
    } else {
        // Reset streak
        $newStreak = 1;
    }
    
    $bestStreak = max($newStreak, $stats['streak_best']);
    
    $stmt = $db->prepare("
        UPDATE user_stats 
        SET streak_current = ?, streak_best = ?, streak_last_date = ? 
        WHERE user_id = ?
    ");
    $stmt->execute([$newStreak, $bestStreak, $today, $userId]);
}
?>
