<?php
/**
 * Anonymous Sync API
 * Auto backup & restore using Device ID (no login required)
 */

require_once '../config.php';

function handleAnonymousSync($action, $method) {
    switch ($action) {
        case 'register':
            if ($method === 'POST') registerDevice();
            break;
        case 'check':
            if ($method === 'POST') checkDevice();
            break;
        case 'sync':
            if ($method === 'POST') syncData();
            break;
        case 'restore':
            if ($method === 'POST') restoreData();
            break;
        case 'get_recovery_code':
            if ($method === 'GET') getRecoveryCode();
            break;
        case 'restore_by_code':
            if ($method === 'POST') restoreByCode();
            break;
        default:
            Response::error('Invalid action', 400);
    }
}

/**
 * Register new device (first time setup)
 */
function registerDevice() {
    $db = Database::getInstance()->getConnection();
    
    $deviceId = Request::post('device_id');
    $deviceName = Request::post('device_name', 'Unknown');
    $platform = Request::post('platform', 'android');
    $userName = Request::post('user_name', 'Student');
    $userClass = Request::post('user_class', 5);
    
    if (!$deviceId) {
        Response::error('Device ID required', 400);
    }
    
    // Check if device already exists
    $stmt = $db->prepare("
        SELECT id, recovery_code, user_id FROM anonymous_devices WHERE device_id = ?
    ");
    $stmt->execute([$deviceId]);
    $existing = $stmt->fetch();
    
    if ($existing) {
        // Device exists - return existing data
        Response::success([
            'is_new' => false,
            'anonymous_id' => $existing['id'],
            'recovery_code' => $existing['recovery_code'],
            'user_id' => $existing['user_id'],
            'message' => 'Device already registered, data available'
        ]);
        return;
    }
    
    // Generate unique recovery code (6 chars alphanumeric)
    $recoveryCode = strtoupper(substr(md5(uniqid(rand(), true)), 0, 6));
    
    // Create anonymous user in users table
    $stmt = $db->prepare("
        INSERT INTO users (username, name, class_grade, password, created_at)
        VALUES (?, ?, ?, ?, NOW())
    ");
    $anonymousUsername = 'anon_' . $deviceId;
    $stmt->execute([$anonymousUsername, $userName, $userClass, 'anonymous']);
    $userId = $db->lastInsertId();
    
    // Create user stats
    $db->prepare("
        INSERT INTO user_stats (user_id, coins, xp, level, streak_current)
        VALUES (?, 100, 0, 1, 0)
    ")->execute([$userId]);
    
    // Register device
    $stmt = $db->prepare("
        INSERT INTO anonymous_devices 
        (device_id, device_name, platform, user_id, recovery_code, created_at)
        VALUES (?, ?, ?, ?, ?, NOW())
    ");
    $stmt->execute([$deviceId, $deviceName, $platform, $userId, $recoveryCode]);
    $anonymousId = $db->lastInsertId();
    
    Response::success([
        'is_new' => true,
        'anonymous_id' => $anonymousId,
        'user_id' => $userId,
        'recovery_code' => $recoveryCode,
        'message' => 'Device registered successfully. Save your recovery code!'
    ]);
}

/**
 * Check if device exists and has data
 */
function checkDevice() {
    $db = Database::getInstance()->getConnection();
    
    $deviceId = Request::post('device_id');
    
    if (!$deviceId) {
        Response::error('Device ID required', 400);
    }
    
    $stmt = $db->prepare("
        SELECT ad.*, u.name, u.class_grade, us.coins, us.xp, us.level, us.streak_current
        FROM anonymous_devices ad
        JOIN users u ON ad.user_id = u.id
        JOIN user_stats us ON u.id = us.user_id
        WHERE ad.device_id = ?
    ");
    $stmt->execute([$deviceId]);
    $device = $stmt->fetch();
    
    if ($device) {
        Response::success([
            'exists' => true,
            'user_id' => $device['user_id'],
            'user_name' => $device['name'],
            'user_class' => $device['class_grade'],
            'coins' => $device['coins'],
            'xp' => $device['xp'],
            'level' => $device['level'],
            'streak' => $device['streak_current'],
            'recovery_code' => $device['recovery_code'],
            'last_sync' => $device['last_sync']
        ]);
    } else {
        Response::success([
            'exists' => false,
            'message' => 'Device not registered'
        ]);
    }
}

/**
 * Sync local data to server
 */
function syncData() {
    $db = Database::getInstance()->getConnection();
    
    $deviceId = Request::post('device_id');
    $coins = Request::post('coins', 0);
    $xp = Request::post('xp', 0);
    $level = Request::post('level', 1);
    $streak = Request::post('streak', 0);
    $gamesPlayed = Request::post('games_played', 0);
    $userName = Request::post('user_name');
    $userClass = Request::post('user_class');
    $achievements = Request::post('achievements', '[]');
    $gameProgress = Request::post('game_progress', '[]');
    
    if (!$deviceId) {
        Response::error('Device ID required', 400);
    }
    
    // Get device
    $stmt = $db->prepare("SELECT user_id FROM anonymous_devices WHERE device_id = ?");
    $stmt->execute([$deviceId]);
    $device = $stmt->fetch();
    
    if (!$device) {
        Response::error('Device not registered', 404);
    }
    
    $userId = $device['user_id'];
    
    // Update user stats (use MAX to never lose progress)
    $db->prepare("
        UPDATE user_stats SET 
            coins = GREATEST(coins, ?),
            xp = GREATEST(xp, ?),
            level = GREATEST(level, ?),
            streak_current = GREATEST(streak_current, ?),
            total_games_played = GREATEST(total_games_played, ?)
        WHERE user_id = ?
    ")->execute([$coins, $xp, $level, $streak, $gamesPlayed, $userId]);
    
    // Update user name/class if provided
    if ($userName || $userClass) {
        $db->prepare("
            UPDATE users SET name = COALESCE(?, name), class_grade = COALESCE(?, class_grade)
            WHERE id = ?
        ")->execute([$userName, $userClass, $userId]);
    }
    
    // Store progress data as JSON
    $db->prepare("
        UPDATE anonymous_devices SET 
            last_sync = NOW(),
            sync_data = ?
        WHERE device_id = ?
    ")->execute([json_encode([
        'achievements' => $achievements,
        'game_progress' => $gameProgress,
        'synced_at' => time()
    ]), $deviceId]);
    
    Response::success([
        'synced' => true,
        'timestamp' => time()
    ], 'Data synced successfully');
}

/**
 * Restore data from server to device
 */
function restoreData() {
    $db = Database::getInstance()->getConnection();
    
    $deviceId = Request::post('device_id');
    
    if (!$deviceId) {
        Response::error('Device ID required', 400);
    }
    
    $stmt = $db->prepare("
        SELECT ad.*, u.name, u.class_grade, us.*
        FROM anonymous_devices ad
        JOIN users u ON ad.user_id = u.id
        JOIN user_stats us ON u.id = us.user_id
        WHERE ad.device_id = ?
    ");
    $stmt->execute([$deviceId]);
    $data = $stmt->fetch();
    
    if (!$data) {
        Response::error('No data found for this device', 404);
    }
    
    // Get achievements
    $stmt = $db->prepare("
        SELECT a.code, a.name, a.emoji FROM user_achievements ua
        JOIN achievements a ON ua.achievement_id = a.id
        WHERE ua.user_id = ?
    ");
    $stmt->execute([$data['user_id']]);
    $achievements = $stmt->fetchAll();
    
    // Get game progress
    $stmt = $db->prepare("
        SELECT * FROM game_progress WHERE user_id = ?
    ");
    $stmt->execute([$data['user_id']]);
    $gameProgress = $stmt->fetchAll();
    
    $syncData = json_decode($data['sync_data'] ?? '{}', true);
    
    Response::success([
        'user_name' => $data['name'],
        'user_class' => $data['class_grade'],
        'coins' => $data['coins'],
        'xp' => $data['xp'],
        'level' => $data['level'],
        'streak' => $data['streak_current'],
        'streak_best' => $data['streak_best'],
        'games_played' => $data['total_games_played'],
        'correct_answers' => $data['total_correct_answers'],
        'achievements' => $achievements,
        'game_progress' => $gameProgress,
        'extra_data' => $syncData,
        'recovery_code' => $data['recovery_code'],
        'last_sync' => $data['last_sync']
    ], 'Data restored successfully');
}

/**
 * Get recovery code for device
 */
function getRecoveryCode() {
    $db = Database::getInstance()->getConnection();
    
    $deviceId = Request::get('device_id');
    
    $stmt = $db->prepare("SELECT recovery_code FROM anonymous_devices WHERE device_id = ?");
    $stmt->execute([$deviceId]);
    $device = $stmt->fetch();
    
    if ($device) {
        Response::success(['recovery_code' => $device['recovery_code']]);
    } else {
        Response::error('Device not found', 404);
    }
}

/**
 * Restore data using recovery code (different device)
 */
function restoreByCode() {
    $db = Database::getInstance()->getConnection();
    
    $recoveryCode = strtoupper(Request::post('recovery_code'));
    $newDeviceId = Request::post('device_id');
    
    if (!$recoveryCode || !$newDeviceId) {
        Response::error('Recovery code and device ID required', 400);
    }
    
    // Find original device by recovery code
    $stmt = $db->prepare("
        SELECT ad.*, u.name, u.class_grade, us.*
        FROM anonymous_devices ad
        JOIN users u ON ad.user_id = u.id
        JOIN user_stats us ON u.id = us.user_id
        WHERE ad.recovery_code = ?
    ");
    $stmt->execute([$recoveryCode]);
    $data = $stmt->fetch();
    
    if (!$data) {
        Response::error('Invalid recovery code', 404);
    }
    
    // Link new device to same user
    $stmt = $db->prepare("
        INSERT INTO anonymous_devices 
        (device_id, device_name, platform, user_id, recovery_code, created_at)
        VALUES (?, 'Restored Device', 'unknown', ?, ?, NOW())
        ON DUPLICATE KEY UPDATE user_id = VALUES(user_id)
    ");
    $stmt->execute([$newDeviceId, $data['user_id'], $recoveryCode]);
    
    Response::success([
        'restored' => true,
        'user_name' => $data['name'],
        'user_class' => $data['class_grade'],
        'coins' => $data['coins'],
        'xp' => $data['xp'],
        'level' => $data['level'],
        'streak' => $data['streak_current'],
        'message' => 'Account restored to new device!'
    ]);
}
?>
